<?php

namespace App\Http\Controllers;

use App\Models\Message;
use Illuminate\Http\Request;

class ChatController extends Controller
{
    // Get list of users to chat with (with unread counts and last message)
    public function getContacts()
    {
        $currentUserId = auth()->id();
        
        // Get all users except current user
        $users = \App\Models\User::where('id', '!=', $currentUserId)
            ->select('id', 'name', 'email', 'role_id')
            ->get();
        
        // Add unread count and last message for each user
        $users->map(function ($user) use ($currentUserId) {
            // Count unread messages from this user to current user
            $user->unread_count = Message::where('user_id', $user->id)
                ->where('receiver_id', $currentUserId)
                ->whereNull('read_at')
                ->count();
            
            // Get last message between users
            $lastMessage = Message::where(function ($query) use ($currentUserId, $user) {
                $query->where('user_id', $currentUserId)
                      ->where('receiver_id', $user->id);
            })->orWhere(function ($query) use ($currentUserId, $user) {
                $query->where('user_id', $user->id)
                      ->where('receiver_id', $currentUserId);
            })
            ->latest()
            ->first();
            
            $user->last_message = $lastMessage ? [
                'message' => $lastMessage->message,
                'created_at' => $lastMessage->created_at,
                'is_mine' => $lastMessage->user_id === $currentUserId
            ] : null;
            
            return $user;
        });
            
        return response()->json($users);
    }

    // Get messages between current user and specific user (auto-mark as read)
    public function getMessages($userId)
    {
        $currentUserId = auth()->id();

        // Mark all unread messages from this user as read
        Message::where('user_id', $userId)
            ->where('receiver_id', $currentUserId)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        // Fetch all messages
        $messages = Message::where(function ($query) use ($currentUserId, $userId) {
            $query->where('user_id', $currentUserId)
                  ->where('receiver_id', $userId);
        })->orWhere(function ($query) use ($currentUserId, $userId) {
            $query->where('user_id', $userId)
                  ->where('receiver_id', $currentUserId);
        })
        ->orderBy('created_at', 'asc')
        ->get();

        return response()->json($messages);
    }

    // Send a message to a specific user
    public function sendMessage(Request $request) 
    {
        $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'message' => 'required|string',
        ]);

        $message = Message::create([
            'user_id' => auth()->id(),
            'receiver_id' => $request->receiver_id,
            'message' => $request->message,
            'is_admin_reply' => false, 
        ]);

        return response()->json($message, 201);
    }

    // Get total unread message count for current user
    public function getUnreadCount()
    {
        $currentUserId = auth()->id();
        
        $unreadCount = Message::where('receiver_id', $currentUserId)
            ->whereNull('read_at')
            ->count();
        
        return response()->json(['unread_count' => $unreadCount]);
    }

    // Mark all messages from a specific user as read
    public function markAsRead($userId)
    {
        $currentUserId = auth()->id();
        
        Message::where('user_id', $userId)
            ->where('receiver_id', $currentUserId)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);
        
        return response()->json(['success' => true]);
    }
}
