<?php

namespace App\Http\Controllers;

use App\Models\Business;
use App\Models\Lead;
use App\Models\Project;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function summary(Request $request)
    {
        $user = $request->user();

        $totalBusinesses = 0;
        $totalLeads = 0;
        $totalProjects = 0;
        
        // Calculate last week's counts for comparison
        $lastWeekBusinesses = 0;
        $lastWeekLeads = 0;
        $lastWeekProjects = 0;
        
        $oneWeekAgo = now()->subWeek();

        // 1. Investor Leads: Count leads added or assigned to current user (for Team/Head)
        // Admin sees all
        if ($user->isAdmin()) {
            $totalLeads = Lead::count();
            $lastWeekLeads = Lead::where('created_at', '<=', $oneWeekAgo)->count();
        } elseif ($user->isHead()) {
             // Head sees leads created by them, or assigned to them/their team, or created by their team (if applicable logic exists, usually assignment is key)
             // Requirement: "assigned to team that count total not overall"
             $teamMemberIds = $user->members->pluck('id')->toArray();
             $teamMemberIds[] = $user->id;

             $totalLeads = Lead::where(function($q) use ($teamMemberIds) {
                 $q->whereIn('created_by', $teamMemberIds)
                   ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                       $sq->whereIn('users.id', $teamMemberIds);
                   });
             })->count();

             $lastWeekLeads = Lead::where(function($q) use ($teamMemberIds) {
                 $q->whereIn('created_by', $teamMemberIds)
                   ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                       $sq->whereIn('users.id', $teamMemberIds);
                   });
             })->where('created_at', '<=', $oneWeekAgo)->count();

        } elseif ($user->isTeamMember()) {
            // Team Member: Added or assigned to current user
            $totalLeads = Lead::where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            })->count();

            $lastWeekLeads = Lead::where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            })->where('created_at', '<=', $oneWeekAgo)->count();
        }

        // 2. Business Leads: Same logic as Investor Leads
        if ($user->isAdmin()) {
            $totalBusinesses = Business::count();
            $lastWeekBusinesses = Business::where('created_at', '<=', $oneWeekAgo)->count();
        } elseif ($user->isHead()) {
             $teamMemberIds = $user->members->pluck('id')->toArray();
             $teamMemberIds[] = $user->id;

             $totalBusinesses = Business::where(function($q) use ($teamMemberIds) {
                 $q->whereIn('created_by', $teamMemberIds)
                   ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                       $sq->whereIn('users.id', $teamMemberIds);
                   });
             })->count();

             $lastWeekBusinesses = Business::where(function($q) use ($teamMemberIds) {
                 $q->whereIn('created_by', $teamMemberIds)
                   ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                       $sq->whereIn('users.id', $teamMemberIds);
                   });
             })->where('created_at', '<=', $oneWeekAgo)->count();

        } elseif ($user->isTeamMember()) {
            $totalBusinesses = Business::where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            })->count();

            $lastWeekBusinesses = Business::where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            })->where('created_at', '<=', $oneWeekAgo)->count();
        }

        // 3. Projects: Total Projects in CRM (Global Count)
        $totalProjects = Project::count();
        $totalOngoingProjects = Project::where('status', 'ongoing')->count();
        $lastWeekProjects = Project::where('created_at', '<=', $oneWeekAgo)->count();
        
        // Calculate weekly growth
        $businessGrowth = $totalBusinesses - $lastWeekBusinesses;
        $leadGrowth = $totalLeads - $lastWeekLeads;
        $projectGrowth = $totalProjects - $lastWeekProjects;

        return response()->json([
            'total_investors' => $totalLeads,
            'total_businesses' => $totalBusinesses,
            'total_projects' => $totalProjects,
            'total_ongoing_projects' => $totalOngoingProjects,
            'business_growth' => $businessGrowth,
            'lead_growth' => $leadGrowth,
            'project_growth' => $projectGrowth,
        ]);
    }

    public function allProjects(Request $request)
    {
        // Redirecting logic to ProjectController with scope=global is preferred, 
        // but likely the frontend just calls this endpoint.
        // For compliance with "Total Projects mean All Total Project In CRM... open list with... Added By team member",
        // We generally shouldn't return the full list in the dashboard summary call, but maybe this endpoint is used by a separate page?
        // Let's implement global list logic here similar to SearchController if this is used for listing.
        
        $user = $request->user();
        
        // Return ALL projects globally
        $projects = Project::with(['assignedTo', 'creator'])
            ->latest()
            ->get()
            ->map(function ($project) use ($user) {
                $hasAccess = $user->isAdmin() || 
                    $user->isHead() || 
                    $project->created_by == $user->id ||
                    $project->assignedTo->contains($user->id);

                return [
                    'id' => $project->id,
                    'company_name' => $project->company_name,
                    'contact_person' => $project->contact_person,
                    'purpose' => $project->purpose,
                    'added_by' => $project->creator->name ?? 'Unknown',
                    'deadline' => $project->deadline,
                     // Mask sensitive
                    'mobile' => $hasAccess ? $project->mobile : '********',
                    'can_view_details' => $hasAccess,
                    'status' => $project->status
                ];
            });

        return response()->json($projects);
    }

    public function ongoingProjects(Request $request)
    {
        $user = $request->user();
        
        // Revert to Role-Based Scope for Dashboard TABLE
        $query = Project::where('status', 'ongoing')->with(['assignedTo', 'creator']);

        if ($user->isAdmin()) {
            // Admin sees all ongoing projects
        } elseif ($user->isHead()) {
            // Head sees projects assigned to them OR their team members OR created by them/team
            $teamMemberIds = $user->members->pluck('id')->toArray();
            $teamMemberIds[] = $user->id; // Include self
            
            $query->where(function($q) use ($teamMemberIds) {
                $q->whereIn('created_by', $teamMemberIds)
                  ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                      $sq->whereIn('users.id', $teamMemberIds);
                  });
            });
        } elseif ($user->isTeamMember()) {
            // Team Member sees ONLY projects assigned to them OR created by them
            $query->where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            });
        }

        $projects = $query->latest()->limit(10)->get();

        // Transform for display (still masking purely defensive, though they should own these)
        $projects->transform(function ($project) use ($user) {
             // For scoped items, they generally have access, but let's be safe
             $hasAccess = $user->isAdmin() || 
                $user->isHead() || 
                $project->created_by == $user->id ||
                $project->assignedTo->contains($user->id);

             return [
                'id' => $project->id,
                'company_name' => $project->company_name,
                'contact_person' => $project->contact_person,
                'purpose' => $project->purpose,
                'added_by' => $project->creator->name ?? 'Unknown',
                'deadline' => $project->deadline,
                'mobile' => $hasAccess ? $project->mobile : '********',
                'can_view_details' => $hasAccess,
                'status' => $project->status,
                'assigned_to' => $project->assignedTo
             ];
        });

        return response()->json($projects);
    }

    public function assignedLeads(Request $request)
    {
        $user = $request->user();
        
        $query = Lead::with(['assignedTo', 'creator']);

        if ($user->isAdmin()) {
            // Admin sees all leads
            $query->latest()->limit(10);
        } elseif ($user->isHead()) {
            // Head sees leads assigned to them/team OR created by them/team
            $teamMemberIds = $user->members->pluck('id')->toArray();
            $teamMemberIds[] = $user->id; // Include Head
            
            $query->where(function($q) use ($teamMemberIds) {
                $q->whereIn('created_by', $teamMemberIds)
                  ->orWhereHas('assignedTo', function ($sq) use ($teamMemberIds) {
                      $sq->whereIn('users.id', $teamMemberIds);
                  });
            })->latest()->limit(10);

        } elseif ($user->isTeamMember()) {
            // Team Member sees leads assigned to them OR created by them
            $query->where(function($q) use ($user) {
                $q->where('created_by', $user->id)
                  ->orWhereHas('assignedTo', function ($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            })->latest()->limit(10);
        }

        $leads = $query->get();

        if ($user->isTeamMember()) {
            $leads->transform(function ($lead) use ($user) {
                // Determine effective access
                $hasAccess = $lead->created_by == $user->id || $lead->assignedTo->contains($user->id);
                
                if (!$hasAccess) {
                    $lead->email = '********';
                    $lead->mobile = '********';
                }
                return $lead;
            });
        }

        return response()->json($leads);
    }
}
