<?php

namespace App\Http\Controllers;

use App\Models\Document;
use App\Models\Lead;
use App\Models\Project;
use App\Models\Business;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DocumentController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'file' => 'required|file|max:10240', // 10MB
            'documentable_id' => 'required|integer',
            'documentable_type' => 'required|string|in:lead,project,business',
        ]);

        $file = $request->file('file');
        $path = $file->store('documents', 'public');

        $modelClass = match ($request->documentable_type) {
            'lead' => Lead::class,
            'project' => Project::class,
            'business' => Business::class,
        };

        $document = Document::create([
            'name' => $file->getClientOriginalName(),
            'path' => $path,
            'documentable_id' => $request->documentable_id,
            'documentable_type' => $modelClass,
            'uploaded_by' => auth()->id(),
        ]);

        return response()->json($document, 201);
    }

    public function index(Request $request)
    {
        $request->validate([
            'documentable_id' => 'required|integer',
            'documentable_type' => 'required|string|in:lead,project,business',
        ]);

        $modelClass = match ($request->documentable_type) {
            'lead' => Lead::class,
            'project' => Project::class,
            'business' => Business::class,
        };
        
        $documents = Document::where('documentable_type', $modelClass)
            ->where('documentable_id', $request->documentable_id)
            ->with('uploader:id,name')
            ->orderByDesc('created_at')
            ->get();

        return response()->json($documents);
    }



    public function destroy($id)
    {
        $document = Document::findOrFail($id);
        
        // Delete file from storage
        if (Storage::disk('public')->exists($document->path)) {
            Storage::disk('public')->delete($document->path);
        }
        
        // Delete database record
        $document->delete();
        
        return response()->json(['message' => 'Document deleted successfully']);
    }
}
