<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class EmployeeController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        $query = User::with(['role', 'head', 'members']);

        // If user is Head, show themselves and their team members
        if ($user->isHead()) {
            // Get team member IDs from the pivot table
            $teamMemberIds = $user->members->pluck('id')->toArray();
            $teamMemberIds[] = $user->id; // Include the Head themselves
            
            // Show users whose ID is in the list (Head + their team members)
            $query->whereIn('id', $teamMemberIds);
        }

        if ($request->has('role')) {
            $query->whereHas('role', function ($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        $employees = $query->get();
        
        return response()->json([
            'data' => $employees
        ]);
    }

    public function show($id, Request $request)
    {
        $user = $request->user();
        $employee = User::with(['role', 'head', 'members'])->findOrFail($id);

        if ($user->isHead()) {
            // Head can only view their own team members or themselves
            $isMyTeam = $employee->head->contains($user->id);
            if (!$isMyTeam && $employee->id !== $user->id) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        }

        return response()->json($employee);
    }


    public function heads()
    {
        // Helper to get all Heads for dropdowns
        $heads = User::whereHas('role', function ($q) {
            $q->where('name', 'head');
        })->get();

        return response()->json($heads);
    }

    public function getTeamMembers(Request $request)
    {
        $currentUser = $request->user();
        
        if ($currentUser->isAdmin()) {
            // Admin: fetch all Heads and Team members (exclude Admin)
            $users = User::with('role')
                ->whereHas('role', function ($q) {
                    $q->whereIn('name', ['head', 'team']);
                })
                ->get();
        } elseif ($currentUser->isHead()) {
            // Head: fetch all Team members only (exclude Head and Admin)
            $users = User::with('role')
                ->whereHas('role', function ($q) {
                    $q->where('name', 'team');
                })
                ->get();
        } else {
            // Team members cannot assign
            $users = [];
        }

        return response()->json($users);
    }

    public function getAssignableUsers(Request $request)
    {
        $currentUser = $request->user();
        
        // Start with base query for Heads and Team Members
        $query = User::with('role')
            ->whereHas('role', function ($q) {
                $q->whereIn('name', ['head', 'team']);
            })
            ->select('id', 'name', 'email', 'role_id');

        // If Head, show ONLY their team members (not themselves)
        if ($currentUser->isHead()) {
            $teamMemberIds = $currentUser->members->pluck('id')->toArray();
            
            // Only show team members, exclude the Head themselves
            $query->whereIn('id', $teamMemberIds);
        } elseif ($currentUser->isTeamMember()) {
            // Team Member sees:
            // 1. Their Head
            // 2. Their Peers (members of the same Head)
            
            $head = $currentUser->head()->first();
            
            if ($head) {
                // Get all members of the head
                $peerIds = $head->members->pluck('id')->toArray();
                
                // Add Head ID
                $allowedIds = array_merge([$head->id], $peerIds);
                
                $query->whereIn('id', $allowedIds);
                // Force exclude self
                $query->where('id', '!=', $currentUser->id);
            } else {
                // Orphan team member? Allow nothing 
                $query->where('id', 0); 
            }
        }
        // Admin sees all Heads and Team Members (no filter needed)

        $users = $query->get();

        return response()->json($users);
    }

    public function roles(Request $request)
    {
        $currentUser = $request->user();
        $query = Role::query();
        
        if ($currentUser->isAdmin()) {
            // Admin can create Head and Team, but not other Admins
            $query->whereIn('name', ['head', 'team']);
        } elseif ($currentUser->isHead()) {
            // Head can only create Team members
            $query->where('name', 'team');
        }
        
        return response()->json($query->get());
    }

    public function store(Request $request)
    {
        $currentUser = $request->user();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'mobile' => 'required|string|unique:users,mobile',
            'password' => 'required|string|min:6',
            'role_id' => 'required|exists:roles,id',
            'region' => 'nullable|string',
            'head_id' => 'nullable|exists:users,id', // Required if role is team
        ]);

        $role = Role::find($validated['role_id']);
        
        // If current user is Head, they can only create Team Members
        if ($currentUser->isHead() && $role->name !== 'team') {
            return response()->json(['message' => 'Heads can only create Team Members'], 403);
        }
        
        // If current user is Head creating a Team Member, auto-assign them as the head
        if ($currentUser->isHead() && $role->name === 'team') {
            $validated['head_id'] = $currentUser->id;
        }
        
        // Validate head_id for team members (only required if user is Admin)
        if ($role->name === 'team' && empty($validated['head_id']) && $currentUser->isAdmin()) {
            return response()->json(['message' => 'Head is required for Team Members'], 422);
        }

        DB::beginTransaction();
        try {
            $validated['password'] = Hash::make($validated['password']);
            
            $user = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'mobile' => $validated['mobile'],
                'password' => $validated['password'],
                'role_id' => $validated['role_id'],
                'region' => $validated['region'],
            ]);

            // If Team Member, assign to Head
            if ($role->name === 'team' && !empty($validated['head_id'])) {
                $head = User::find($validated['head_id']);
                // Check if head is actually a head
                if (!$head->isHead()) {
                     throw new \Exception("Selected user is not a Head");
                }
                $head->members()->attach($user->id);
            }

            DB::commit();
            return response()->json($user->load(['role', 'head']), 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to create employee: ' . $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|email|unique:users,email,' . $id,
            'mobile' => 'sometimes|required|string|unique:users,mobile,' . $id,
            'password' => 'nullable|string|min:6',
            'role_id' => 'sometimes|required|exists:roles,id',
            'region' => 'nullable|string',
            'head_id' => 'nullable|exists:users,id',
        ]);

        DB::beginTransaction();
        try {
            if (isset($validated['password'])) {
                $validated['password'] = Hash::make($validated['password']);
            }

            $user->update($validated);

            // Update Head assignment if provided
            if (isset($validated['head_id'])) {
                // Remove existing head assignment
                DB::table('team_members')->where('member_id', $user->id)->delete();
                
                // Add new assignment
                $head = User::find($validated['head_id']);
                if ($head && $head->isHead()) {
                    $head->members()->attach($user->id);
                }
            }

            DB::commit();
            return response()->json($user->load(['role', 'head']));

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to update employee'], 500);
        }
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return response()->json(['message' => 'Employee deleted successfully']);
    }
}
