<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\Task;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TaskController extends Controller
{
    public function index($projectId, Request $request)
    {
        $user = $request->user();
        $project = Project::with('assignedTo')->findOrFail($projectId);

        // Check project access
        if ($user->isAdmin()) {
            // Admin has full access
        } elseif ($user->isHead()) {
            // Head can view if project is assigned to them OR their team members
            $teamMemberIds = $user->members->pluck('id')->toArray();
            $teamMemberIds[] = $user->id; // Include self
            
            $hasAccess = $project->assignedTo->whereIn('id', $teamMemberIds)->isNotEmpty();
            
            if (!$hasAccess) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        } elseif ($user->isTeamMember()) {
            // Team Member can view if project is assigned to them
            if (!$project->assignedTo->contains($user->id)) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        }

        $query = $project->tasks()->with(['assignee', 'assignedUsers'])->latest();

        // Filter tasks based on role
        if ($user->isTeamMember()) {
            // Team Member sees ONLY tasks assigned to them (either as primary assignee or in assignedUsers)
            $query->where(function($q) use ($user) {
                $q->where('assigned_to', $user->id)
                  ->orWhereHas('assignedUsers', function($sq) use ($user) {
                      $sq->where('users.id', $user->id);
                  });
            });
        }

        $tasks = $query->get();
        return response()->json($tasks);
    }

    public function store(Request $request, $projectId)
    {
        $user = $request->user();
        if ($user->isTeamMember()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $project = Project::findOrFail($projectId);
        
        // Head check
        if ($user->isHead() && !$project->assignedTo->contains($user->id)) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assigned_to' => 'nullable|exists:users,id', // Keep for backward compatibility
            'user_ids' => 'nullable|array', // New field for multiple assignments
            'user_ids.*' => 'exists:users,id',
            'status' => 'required|in:pending,in-progress,completed',
            'due_date' => 'nullable|date',
        ]);

        // Validate assignments
        $assignedUserIds = $validated['user_ids'] ?? [];
        if (!empty($validated['assigned_to'])) {
            $assignedUserIds[] = $validated['assigned_to'];
        }
        $assignedUserIds = array_unique($assignedUserIds);

        if (!empty($assignedUserIds)) {
            $assignedUsers = \App\Models\User::whereIn('id', $assignedUserIds)->with('role')->get();
            
            foreach ($assignedUsers as $assignedUser) {
                // 1. Prevent assigning to Admins
                if ($assignedUser->isAdmin()) {
                    return response()->json(['message' => "Cannot assign tasks to Admin users ({$assignedUser->name})"], 422);
                }

                // 2. Head can only assign to their team members (not themselves)
                if ($user->isHead()) {
                    // Check if assigned user is in head's team
                    $isMyTeam = $assignedUser->head->contains($user->id);
                    if (!$isMyTeam) {
                        return response()->json(['message' => "You can only assign tasks to your team members ({$assignedUser->name})"], 403);
                    }
                }
            }
        }

        $task = $project->tasks()->create([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'assigned_to' => $validated['assigned_to'] ?? null,
            'status' => $validated['status'],
            'due_date' => $validated['due_date'] ?? null,
        ]);

        // Attach multiple users if provided
        if (!empty($validated['user_ids'])) {
            $task->assignedUsers()->attach($validated['user_ids']);
        }
        
        return response()->json(['message' => 'Task created successfully', 'task' => $task->load(['assignee', 'assignedUsers'])], 201);
    }

    public function update(Request $request, $id)
    {
        $task = Task::findOrFail($id);
        $user = $request->user();

        // Team member can only update status of assigned tasks
        if ($user->isTeamMember()) {
            if ($task->assigned_to !== $user->id) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
            
            $validated = $request->validate([
                'status' => 'required|in:pending,in-progress,completed',
            ]);
            
            $task->update(['status' => $validated['status']]);
            return response()->json(['message' => 'Task status updated', 'task' => $task->load('assignee')]);
        }

        // Head check
        if ($user->isHead()) {
            $project = $task->project;
            if (!$project->assignedTo->contains($user->id)) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        }

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'assigned_to' => 'nullable|exists:users,id',
            'user_ids' => 'nullable|array',
            'user_ids.*' => 'exists:users,id',
            'status' => 'sometimes|required|in:pending,in-progress,completed',
            'due_date' => 'nullable|date',
        ]);

        // Validate assignments
        $assignedUserIds = $validated['user_ids'] ?? [];
        if (!empty($validated['assigned_to'])) {
            $assignedUserIds[] = $validated['assigned_to'];
        }
        $assignedUserIds = array_unique($assignedUserIds);

        if (!empty($assignedUserIds)) {
            $assignedUsers = \App\Models\User::whereIn('id', $assignedUserIds)->with('role')->get();
            
            foreach ($assignedUsers as $assignedUser) {
                // 1. Prevent assigning to Admins
                if ($assignedUser->isAdmin()) {
                    return response()->json(['message' => "Cannot assign tasks to Admin users ({$assignedUser->name})"], 422);
                }

                // 2. Head can only assign to their team members (not themselves)
                if ($user->isHead()) {
                    // Check if assigned user is in head's team
                    $isMyTeam = $assignedUser->head->contains($user->id);
                    if (!$isMyTeam) {
                        return response()->json(['message' => "You can only assign tasks to your team members ({$assignedUser->name})"], 403);
                    }
                }
            }
        }

        $task->update([
            'title' => $validated['title'] ?? $task->title,
            'description' => $validated['description'] ?? $task->description,
            'assigned_to' => $validated['assigned_to'] ?? $task->assigned_to,
            'status' => $validated['status'] ?? $task->status,
            'due_date' => $validated['due_date'] ?? $task->due_date,
        ]);

        // Sync multiple users if provided
        if (isset($validated['user_ids'])) {
            $task->assignedUsers()->sync($validated['user_ids']);
        }

        return response()->json(['message' => 'Task updated successfully', 'task' => $task->load(['assignee', 'assignedUsers'])]);
    }

    public function destroy(Request $request, $id)
    {
        $user = $request->user();
        if ($user->isTeamMember()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $task = Task::findOrFail($id);
        
        // Head check
        if ($user->isHead()) {
            $project = $task->project;
            if (!$project->assignedTo->contains($user->id)) {
                return response()->json(['message' => 'Unauthorized'], 403);
            }
        }

        $task->delete();

        return response()->json(['message' => 'Task deleted successfully']);
    }
}
