<?php

namespace Database\Seeders;

use App\Models\Business;
use App\Models\Lead;
use App\Models\Project;
use App\Models\Role;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create roles
        $adminRole = Role::create(['name' => 'admin', 'permissions' => ['all']]);
        $headRole = Role::create(['name' => 'head', 'permissions' => ['business', 'leads', 'projects']]);
        $teamRole = Role::create(['name' => 'team', 'permissions' => ['business', 'leads', 'projects']]);

        // Create Users
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@valmiki.com',
            'mobile' => '9999999999',
            'password' => Hash::make('password'),
            'role_id' => $adminRole->id,
            'region' => 'All',
        ]);

        $head = User::create([
            'name' => 'Head Manager',
            'email' => 'head@valmiki.com',
            'mobile' => '9999999998',
            'password' => Hash::make('password'),
            'role_id' => $headRole->id,
            'region' => 'North',
        ]);

        $teamUser = User::create([
            'name' => 'Team Member',
            'email' => 'team@valmiki.com',
            'mobile' => '9999999997',
            'password' => Hash::make('password'),
            'role_id' => $teamRole->id,
            'region' => 'North',
        ]);

        // Assign Team Member to Head (Hierarchy)
        $head->members()->attach($teamUser->id);

        // Seed Business Leads
        $purposes = [
            'Initial Public Offer (IPO) & Rights Issue',
            'Private Equity & Pre IPO Fund Raise',
            'QIPs & Preferential Allotments',
            'Mergers, Acquisitions & Corporate Restructuring',
            'Buy-Side, Sell-Side, Takeover & Open Offer Advisory',
            'Corporate Advisory & Corporate Valuation',
            'Debt Syndication & Structured Debt Solutions'
        ];

        for ($i = 1; $i <= 15; $i++) {
            $business = Business::create([
                'person_name' => "Business Person $i",
                'company_name' => "Business Company $i",
                'email' => "business$i@example.com",
                'mobile' => "98765432" . str_pad($i, 2, '0', STR_PAD_LEFT),
                'purpose' => $purposes[array_rand($purposes)],
                'size' => rand(1, 100) . ' Cr',
                'notes' => 'Interested in services',
                'next_followup_date' => now()->addDays(rand(1, 30)),
                'created_by' => $admin->id,
            ]);
            // Assign to Head
            $business->assignedTo()->attach($head->id);
        }

        // Seed Investor Leads
        for ($i = 1; $i <= 25; $i++) {
            $isAccepted = rand(0, 1);
            $lead = Lead::create([
                'person_name' => "Investor Person $i",
                'company_name' => "Investor Company $i",
                'email' => "investor$i@example.com",
                'mobile' => "87654321" . str_pad($i, 2, '0', STR_PAD_LEFT),
                'size' => rand(1, 50) . ' Cr',
                'notes' => 'Potential investor',
                'lead_quality' => rand(0, 1) ? 'priority' : 'general',
                'purpose' => $purposes[array_rand($purposes)],
                'status' => ['new', 'follow-up', 'lost', 'open'][rand(0, 3)],
                'next_followup_date' => now()->addDays(rand(1, 30)),
                'is_imported' => false,
                'is_accepted' => $isAccepted,
                'created_by' => $head->id,
            ]);
            
            // Assign to Head
            $lead->assignedTo()->attach($head->id, [
                'accepted_at' => $isAccepted ? now() : null,
            ]);
        }

        // Seed Projects
        for ($i = 1; $i <= 8; $i++) {
            $project = Project::create([
                'company_name' => "Project Company $i",
                'contact_person' => "Project Contact $i",
                'purpose' => $purposes[array_rand($purposes)],
                'deadline' => now()->addMonths(rand(1, 6)),
                'status' => rand(0, 1) ? 'ongoing' : 'completed',
                'notes' => 'Project notes',
                'created_by' => $admin->id,
            ]);
            // Assign to Head
            $project->assignedTo()->attach($head->id);

            // Seed Tasks for Project
            for ($j = 1; $j <= 3; $j++) {
                $project->tasks()->create([
                    'title' => "Task $j for Project $i",
                    'description' => 'Task description',
                    'assigned_to' => $teamUser->id,
                    'status' => ['pending', 'in-progress', 'completed'][rand(0, 2)],
                    'due_date' => now()->addDays(rand(1, 14)),
                ]);
            }
        }

        $this->command->info('Database seeded successfully!');
        $this->command->info('Admin: admin@valmiki.com / password');
        $this->command->info('Head: head@valmiki.com / password');
        $this->command->info('Team: team@valmiki.com / password');
    }
}
