<?php

use App\Http\Controllers\AuthController;
use App\Http\Controllers\BusinessController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\LeadController;
use App\Http\Controllers\ProjectController;
use App\Http\Controllers\EmployeeController;
use App\Http\Controllers\SearchController;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\ToDoController;
use App\Http\Controllers\ChatController;
use App\Http\Controllers\DocumentController;
use Illuminate\Support\Facades\Route;
use App\Models\User;

// Public routes
Route::post('/auth/login', [AuthController::class, 'login'])->name('login');

// Debug route to check hierarchy
Route::get('/debug/hierarchy/{userId}', function ($userId) {
    $user = User::with(['role', 'members', 'head'])->find($userId);
    
    if (!$user) {
        return response()->json(['error' => 'User not found'], 404);
    }
    
    $teamMemberIds = [];
    if ($user->isHead()) {
        $teamMemberIds = $user->members->pluck('id')->toArray();
    }
    
    return response()->json([
        'user' => [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'role' => $user->role->name,
        ],
        'is_admin' => $user->isAdmin(),
        'is_head' => $user->isHead(),
        'is_team_member' => $user->isTeamMember(),
        'team_members' => $user->members->map(fn($m) => [
            'id' => $m->id,
            'name' => $m->name,
            'email' => $m->email,
        ]),
        'team_member_ids' => $teamMemberIds,
        'heads' => $user->head->map(fn($h) => [
            'id' => $h->id,
            'name' => $h->name,
            'email' => $h->email,
        ]),
    ]);
});

// Protected routes
Route::middleware('auth:sanctum')->group(function () {
    // Auth routes
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    Route::get('/auth/me', [AuthController::class, 'me']);

    // Employee routes
    Route::get('/employees', [EmployeeController::class, 'index'])
        ->middleware('role:admin,head');
    Route::get('/employees/{employee}', [EmployeeController::class, 'show'])
        ->middleware('role:admin,head');
    Route::post('/employees', [EmployeeController::class, 'store'])
        ->middleware('role:admin,head');
    Route::put('/employees/{employee}', [EmployeeController::class, 'update'])
        ->middleware('role:admin');
    Route::delete('/employees/{employee}', [EmployeeController::class, 'destroy'])
        ->middleware('role:admin');
    Route::get('/heads', [EmployeeController::class, 'heads'])->middleware('role:admin,head');
    Route::get('/team-members', [EmployeeController::class, 'getTeamMembers'])->middleware('role:admin,head');
    Route::get('/roles', [EmployeeController::class, 'roles'])->middleware('role:admin,head');
    Route::get('/users/assignable', [EmployeeController::class, 'getAssignableUsers'])
        ->middleware('role:admin,head,team');

    // Dashboard routes
    Route::get('/dashboard/summary', [DashboardController::class, 'summary']);
    Route::get('/dashboard/all-projects', [DashboardController::class, 'allProjects']);
    Route::get('/dashboard/ongoing-projects', [DashboardController::class, 'ongoingProjects']);
    Route::get('/dashboard/assigned-leads', [DashboardController::class, 'assignedLeads']);

    // Global search
    Route::get('/search', [SearchController::class, 'search']);

    // Business routes
    Route::post('/businesses/import', [BusinessController::class, 'import'])
        ->middleware('role:admin,head')
        ->middleware('module.access:business');
    Route::apiResource('businesses', BusinessController::class)
        ->middleware('module.access:business');
    Route::post('/businesses/{id}/convert', [BusinessController::class, 'convertToProject'])
        ->middleware('module.access:business');

    // Lead routes
    Route::apiResource('leads', LeadController::class)
        ->middleware('module.access:leads');
    Route::post('/leads/import', [LeadController::class, 'import'])
        ->middleware('role:admin,head')
        ->middleware('module.access:leads');
    Route::get('/leads/newly-assigned', [LeadController::class, 'newlyAssigned'])
        ->middleware('module.access:leads');
    Route::post('/leads/{id}/accept', [LeadController::class, 'accept'])
        ->middleware('module.access:leads');
    Route::post('/leads/{id}/reject', [LeadController::class, 'reject'])
        ->middleware('module.access:leads');
    Route::get('/leads/priority/list', [LeadController::class, 'priorityLeads'])
        ->middleware('module.access:leads');
    Route::get('/leads/rejected/list', [LeadController::class, 'rejectedLeads'])
        ->middleware('role:admin,head')
        ->middleware('module.access:leads');
    Route::post('/leads/{id}/convert', [LeadController::class, 'convertToProject'])
        ->middleware('module.access:leads');

    // To-Do Logic
    Route::get('/todo', [ToDoController::class, 'index']);

    // Chat Logic
    Route::get('/chat/contacts', [ChatController::class, 'getContacts']);
    Route::get('/chat/messages/{userId}', [ChatController::class, 'getMessages']);
    Route::post('/chat/send', [ChatController::class, 'sendMessage']);
    Route::get('/chat/unread-count', [ChatController::class, 'getUnreadCount']);
    Route::post('/chat/mark-read/{userId}', [ChatController::class, 'markAsRead']);

    // Documents
    Route::get('/documents', [DocumentController::class, 'index']);
    Route::post('/documents', [DocumentController::class, 'store']);
    Route::delete('/documents/{id}', [DocumentController::class, 'destroy']);


    // Project routes (Admin and Head only)
    Route::post('/projects/import', [ProjectController::class, 'import'])
        ->middleware('module.access:projects');
    Route::apiResource('projects', ProjectController::class)
        ->middleware('module.access:projects');
    
    // Task routes
    Route::get('/projects/{projectId}/tasks', [TaskController::class, 'index'])
        ->middleware('module.access:projects');
    Route::post('/projects/{projectId}/tasks', [TaskController::class, 'store'])
        ->middleware('module.access:projects');
    Route::put('/tasks/{id}', [TaskController::class, 'update'])
        ->middleware('module.access:projects');
    Route::delete('/tasks/{id}', [TaskController::class, 'destroy'])
        ->middleware('module.access:projects');
});
